/****************************************************************************
*  LPC82x RC5 decoder demo program for LPC 824 board
*
*  1. Use SCT timer to capture and decode RC5 messages.
*  2. Use USART to print received RC5 frames to PC terminal
*****************************************************************************/
#include "board.h"
#include "sct_fsm.h"


/* Define expected signal characteristics */
#define PULSE_FREQUENCY			36000		/* Burst pulse frequency in Hz */
#define PULSE_TICKS				111			/* Ticks per 36 kHz pulse */
#define PULSE_DUTY_CYCLE		25			/* Duty cycle in % */

/* Derived constants */
#define SCT_PRESCALER			(SystemCoreClock / (PULSE_FREQUENCY * 100u))
#define SCT_MATCH_DUTY_CYCLE	(PULSE_TICKS - 1 - (PULSE_DUTY_CYCLE * PULSE_TICKS) / 100u)
#define SCT_MATCH_PULSE_LENGTH	(PULSE_TICKS - 1)

#define RC5_BIT_DIVIDER			(SCT_PRESCALER * PULSE_TICKS * 32)



#define PIO_LED			PIO0_12				/* Red LED */
#define PIO_DUMMY_DATA	PIO0_15				/* Dummy RC-5 data pin */


#define RC5_DATA0()      Chip_IOCON_PinSetMode(LPC_IOCON, IOCON_PIO15, PIN_MODE_PULLDN);	// input low (pull down)
#define RC5_DATA1()      Chip_IOCON_PinSetMode(LPC_IOCON, IOCON_PIO15, PIN_MODE_PULLUP); 	// input high (pull up)

void sct_fsm_init (void);


struct
{
	uint32_t data;
	int state;
	volatile int bitsLeft;
} rc5;


void sct_fsm_init(void)
{

    Chip_SCT_Init(LPC_SCT);															// enable the SCT clock

    Chip_SCT_Config(LPC_SCT, SCT_CONFIG_16BIT_COUNTER);								// split timer

    Chip_SCT_SetControl(LPC_SCT, (((SCT_PRESCALER - 1) << SCT_CTRL_L_PRE_L_Pos) & SCT_CTRL_L_PRE_L_Msk));

    Chip_SCT_Output(LPC_SCT, SCT_EVE_0 );                      						// SCT_OUT0 (IR output) is high



    Chip_SCT_SetMatchCountL(LPC_SCT, SCT_MATCH_0, SCT_MATCH_DUTY_CYCLE);			// 75% of 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT, SCT_MATCH_0, SCT_MATCH_DUTY_CYCLE);

    Chip_SCT_SetMatchCountL(LPC_SCT, SCT_MATCH_1, SCT_MATCH_PULSE_LENGTH);			// 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT, SCT_MATCH_1, SCT_MATCH_PULSE_LENGTH);


    Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, ENABLE_ALL_STATES);		// event 0 happens in all states
    Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match 0
                                                                          SCT_HEVENT_L		    |	// HEVENT   [4]     = use L state & match
                                                                          SCT_OUTSEL_L		    |	// OUTSEL   [5]     = select input from IOSEL
                                                                          SCT_IOSEL_RISEIN0	  |	// IOSEL    [9:6]   = select input 0
                                                                          SCT_IOCOND_HIGH		  |	// IOCOND   [11:10] = high level
                                                                          SCT_COMBMODE_AND	  |	// COMBMODE [13:12] = match AND IO condition
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    |	// STATEV   [19:15] = no state change
                                                                          SCT_MATCHMEM		    |	// MATCHMEM [20]    = "equal" to match
                                                                          SCT_DIRECTION		  ));	// DIRECTION[22:21] = direction independent


    Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, ENABLE_ALL_STATES);		// event 1 happens in all states
    Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1	|	// MATCHSEL [3:0]   = related to match 1
                                                                          SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    ));	// STATEV   [19:15] = no state change


    Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_0);				// IR LED low  @ event 0
    Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_1);				// IR LED high @ event 1

    LPC_SCT->LIMIT_L        = (1 << 1);                   				// events 1 is used as counter limit


    LPC_SCT->LIMIT_H        = (1 << 4);                   					// event 4 is used as H counter limit

    NVIC_EnableIRQ(SCT_IRQn);                             					// enable SCT interrupt

    Chip_SCT_ClearControl(LPC_SCT,(SCT_CTRL_HALT_L |SCT_CTRL_HALT_H));      // unhalt both low and high counter

}


/* MRT interrupt handler */
void MRT_IRQHandler (void)
{
	/* Channel 0: Next Rc-5 bit */
	if (Chip_MRT_IntPending(LPC_MRT_CH0)) {
		/* Send next bit if required */
		if (rc5.bitsLeft) {
			if (rc5.state) {
				if (rc5.data & 0x80000000) {
					RC5_DATA1();
				}
				else {
					RC5_DATA0();
				}

				rc5.state = 0;
				--rc5.bitsLeft;
				rc5.data <<= 1;
			}
			else {
				if (rc5.data & 0x80000000) {
					RC5_DATA0();
				}
				else {
					RC5_DATA1();
				}

				rc5.state = 1;
			}
		}
		else {
			/* Inactive */
			RC5_DATA0();
		}

		/* Acknowledge interrupt */
		Chip_MRT_IntClear(LPC_MRT_CH0);
	}
}


void MRT_Init(void)
{
	Chip_MRT_Init();									  		// enable MRT clock

	Chip_MRT_SetInterval(LPC_MRT_CH0, RC5_BIT_DIVIDER|
									  (1U << 31)	 );			//  Bit rate timer & Enable in repeat mode


	Chip_MRT_SetMode(LPC_MRT_CH0, MRT_MODE_REPEAT	);     		// enable + repeated mode
    NVIC_EnableIRQ(MRT_IRQn);                              		// enable Multi-Rate timer irq
}

int main(void)
{

  volatile static int i;

  SystemCoreClockUpdate();
  Board_Init();


  Chip_Clock_EnablePeriphClock(	(CHIP_SYSCTL_CLOCK_T)  ( SYSCTL_CLOCK_SWM 	  | 			// enable SWM clock
                                                      SYSCTL_CLOCK_GPIO 	|			// enable GPIO port clock
                                                      SYSCTL_CLOCK_IOCON	));		    //enable IOCON clock

  Chip_SWM_MovablePinAssign(SWM_SCT_IN0_I, 15);					// CTIN_0 -> PIO0_14 (PWM input)
  Chip_SWM_MovablePinAssign(SWM_SCT_OUT0_O, 12);				// CTOUT_0 -> PIO0_12 (Red LED)



  MRT_Init();
  sct_fsm_init();

    while (1)
    {
    	rc5.data = 0xD4480000;				/* System=20 (CD), Digit 9 */
    	rc5.state = 0;
    	rc5.bitsLeft = 14;
    	while (rc5.bitsLeft);

    	for (i = 0; i < 100000; i++);
   }
}

